<?php

namespace App\Http\Controllers;

use App\Models\Gig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use App\Models\Story;
class GigController extends Controller
{
    // 🟢 Create a new Gig
    public function store(Request $request)
    {
        // Log user information to debug
        Log::info('User Role: ' . Auth::user()->role);

        // Checking the user's role directly
        // if (!in_array(Auth::user()->role, ['Content Creator', 'Admin'])) {
        //     return response()->json(['message' => 'Unauthorized'], 403);
        // }
        if (!Auth::user()->hasRole('Content Creator') && !Auth::user()->hasRole('Admin')) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
	    
            'label'          => 'required|string|max:255',
            'description'    => 'required|string',
            'starting_price' => 'required|numeric|min:0',
            'media'          => 'nullable|file|mimes:jpg,jpeg,png,mp4|max:10240', // 10MB max
        ]);

        $path = $request->file('media') ? $request->file('media')->store('gigs', 'public') : null;

        $gig = Gig::create([
            'user_id'        => Auth::id(),
            'label'          => $request->label,
            'description'    => $request->description,
            'starting_price' => $request->starting_price,
            'media_path'     => $path,
            'rating'         => 0,
            'review'         => 0,
            'is_favorite'    => false,
        ]);

        return response()->json(['message' => 'Gig created successfully', 'gig' => $gig]);
    }


    // 🔵 List all Gigs (for this user or all if admin)
    public function index()
    {
        $user = Auth::user();
        $gigs = $user->role === 'Admin' ? Gig::all() : $user->gigs;

        return response()->json($gigs);
    }

    // 🟡 Show single Gig

    public function show($id)
    {
        $gig = Gig::find($id);

        if (!$gig) {
            Log::info('Gig ID not found: ' . $id);
            return response()->json(['message' => 'Gig not found'], 404);
        }

        return response()->json($gig);
    }





    // 🟠 Update a Gig
    public function update(Request $request, $id)
    {
        $gig = Gig::findOrFail($id);

        // Role check for the owner or admin
        if ($gig->user_id !== Auth::id() && Auth::user()->role !== 'Admin') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'label'          => 'sometimes|required|string|max:255',
            'description'    => 'sometimes|required|string',
            'starting_price' => 'sometimes|required|numeric|min:0',
            'media'          => 'nullable|file|mimes:jpg,jpeg,png,mp4|max:10240',
        ]);

        if ($request->hasFile('media')) {
            if ($gig->media_path) {
                Storage::disk('public')->delete($gig->media_path);
            }
            $gig->media_path = $request->file('media')->store('gigs', 'public');
        }

        $gig->update($request->only(['label', 'description', 'starting_price']));

        return response()->json(['message' => 'Gig updated successfully', 'gig' => $gig]);
    }

    // 🔴 Delete a Gig
    public function destroy($id)
    {
        $gig = Gig::findOrFail($id);

        // Role check for the owner or admin
        if ($gig->user_id !== Auth::id() && Auth::user()->role !== 'Admin') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($gig->media_path) {
            Storage::disk('public')->delete($gig->media_path);
        }

        $gig->delete();
        return response()->json(['message' => 'Gig deleted successfully']);
    }

    // ❤️ Add/Remove Favorite
    public function toggleFavorite($id)
    {
        $gig = Gig::findOrFail($id);
        $gig->is_favorite = !$gig->is_favorite;
        $gig->save();

        return response()->json(['message' => 'Favorite status updated', 'is_favorite' => $gig->is_favorite]);
    }




    public function filterGigs(Request $request)
{
    // Start with the base query
    $query = Gig::query();

    // Check if there's a label filter in the request
    if ($request->has('label')) {
        // You can pass multiple labels as a comma-separated list
        $labels = explode(',', $request->label);
        $query->whereIn('label', $labels); // Filter by labels
    }

    // Apply other filters if needed (e.g., search by description)
    if ($request->has('search')) {
        $query->where('description', 'like', '%' . $request->search . '%');
    }

    // Pagination - default to 10 gigs per page if not specified
    $perPage = $request->get('per_page', 10);
    $gigs = $query->latest()->paginate($perPage);

    return response()->json([
        'message' => 'Filtered gigs with pagination',
        'data' => $gigs
    ]);
}

public function getFilters()
{
    // Get unique labels from the gigs table
    $labels = Gig::distinct()->pluck('label');

    return response()->json([
        'message' => 'Available gig filters',
        'data' => $labels
    ]);
}

public function checkFilterExists(Request $request)
{
    $label = $request->input('label');  // Get label from the request

    // Check if the label exists in the database
    $exists = Gig::where('label', $label)->exists();

    return response()->json([
        'message' => $exists ? 'Filter exists' : 'Filter does not exist',
        'exists' => $exists
    ]);
}


public function buyerDashboard()
{
    // Get all active stories from eligible content creators
    $stories = Story::active()
        ->whereHas('creator', function ($q) {
            $q->where(function ($query) {
                $query->where('instagram_followers', '>=', 1000)
                      ->orWhere('youtube_subscribers', '>=', 1000);
            });
        })
        ->with(['creator.user:id,name']) // to show content creator name
        ->latest()
        ->get()
        ->map(function ($story) {
            return [
                'story_id' => $story->id,
                'creator_name' => $story->creator->user->name,
                'media_url' => $story->media_path ? asset('storage/' . $story->media_path) : null,
                'caption' => $story->caption,
                'expires_at' => $story->expires_at,
            ];
        });

    // Get all gigs and include the creator name
    $gigs = Gig::with('user:id,name') // to show content creator name
        ->latest()
        ->get()
        ->map(function ($gig) {
            return [
                'gig_id' => $gig->id,
                'creator_name' => $gig->user->name,
                'media_url' => $gig->media_url,
                'label' => $gig->label,
                'description' => $gig->description,
                'starting_price' => $gig->starting_price,
            ];
        });

    return response()->json([
        'stories' => $stories,
        'gigs' => $gigs,
    ]);
}



public function showInBuyer($id)
{
    $gig = Gig::with('user:id,name,email')->find($id);

    if (!$gig) {
        return response()->json(['message' => 'Gig not found.'], 404);
    }

    return response()->json([
        'label' => $gig->label,
        'description' => $gig->description,
        'media_url' => $gig->media_url, // This uses the accessor
        'starting_price' => $gig->starting_price,
        'rating' => $gig->rating,
        'review' => $gig->review,
        'created_by' => [
            'name' => $gig->user->name,
            'email' => $gig->user->email,
        ]
    ]);
}




}
