<?php

namespace App\Http\Controllers;

use App\Models\CreatorService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\CreatorProfile;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;


class CreatorServiceController extends Controller
{
    // Create a new service
    // public function store(Request $request)
    // {
    //     $request->validate([
    //         'service_title' => 'required|string|max:255',
    //         'service_price' => 'required|numeric|min:0',
    //     ]);

    //     $creatorProfile = Auth::user()->contentCreator;

    //     if (!$creatorProfile) {
    //         return response()->json(['message' => 'Creator profile not found.'], 404);
    //     }

    //     $service = CreatorService::create([
    //         'creator_profile_id' => $creatorProfile->id,
    //         'service_title' => $request->service_title,
    //         'service_price' => $request->service_price,
    //     ]);

    //     return response()->json(['message' => 'Service created successfully!', 'service' => $service], 201);
    // }

    public function store(Request $request)
{
    // Get the creator profile of the logged-in user
    $creatorProfile = CreatorProfile::where('user_id', auth()->id())->first();

    if (!$creatorProfile) {
        return response()->json(['message' => 'Creator profile not found.'], 404);
    }

    // Proceed to create the service
    $service = $creatorProfile->services()->create([
        'service_title' => $request->service_title,
        'service_price' => $request->service_price,
    ]);

    return response()->json(['message' => 'Service created successfully!', 'service' => $service]);
}

    // List all services for the logged-in creator
    public function index(Request $request)
    {
        // Get the creator profile of the logged-in user
        $creatorProfile = Auth::user()->creatorProfile;

        if (!$creatorProfile) {
            return response()->json(['message' => 'Creator profile not found.'], 404);
        }

        // Load services for the creator profile
        $services = $creatorProfile->services()->get(); // Use ->get() to make sure it's an empty collection if no services

        // Check if no services are found
        if ($services->isEmpty()) {
            return response()->json(['message' => 'No services found for this creator.'], 404);
        }

        return response()->json(['services' => $services]);
    }



    // Update an existing service
    public function update(Request $request, $id)
    {
        $request->validate([
            'service_title' => 'required|string|max:255',
            'service_price' => 'required|numeric|min:0',
        ]);

        $creatorProfile = Auth::user()->creatorProfile;

        if (!$creatorProfile) {
            return response()->json(['message' => 'Creator profile not found.'], 404);
        }

        $service = CreatorService::findOrFail($id);

        // Ensure the service belongs to the authenticated creator
        if ($service->creator_profile_id !== $creatorProfile->id) {
            return response()->json(['message' => 'Unauthorized. You can only update your own services.'], 403);
        }

        $service->update($request->only('service_title', 'service_price'));

        return response()->json(['message' => 'Service updated successfully!', 'service' => $service]);
    }

    // Delete a service
    public function destroy($id)
    {
        $creatorProfile = Auth::user()->creatorProfile;

        if (!$creatorProfile) {
            return response()->json(['message' => 'Creator profile not found.'], 404);
        }

        $service = CreatorService::findOrFail($id);

        // Ensure the service belongs to the authenticated creator
        if ($service->creator_profile_id !== $creatorProfile->id) {
            return response()->json(['message' => 'Unauthorized. You can only delete your own services.'], 403);
        }

        $service->delete();

        return response()->json(['message' => 'Service deleted successfully.']);
    }


        // Search services based on a keyword (used by the buyer)
        public function search(Request $request)
        {
            // Validate the search query input
            $request->validate([
                'keyword' => 'required|string|min:3|max:255', // Ensure the keyword is between 3 and 255 characters
            ]);

            // Sanitize the keyword (important to prevent XSS or SQL injection)
            $keyword = htmlspecialchars($request->keyword, ENT_QUOTES, 'UTF-8');

            // Search for services by matching the keyword in the service title
            $services = CreatorService::where('service_title', 'like', '%' . $keyword . '%')->paginate(10); // Paginate to limit results

            // If no services are found, return a message
            if ($services->isEmpty()) {
                return response()->json(['message' => 'No services found for this keyword.'], 404);
            }

            return response()->json(['services' => $services]);
        }

        // Get the top 5 recommended services based on the most frequently used words in the service titles
        public function getRecommendations()
        {
            // Check if recommendations are cached
            $recommendedServices = Cache::get('recommended_services');

            if (!$recommendedServices) {
                // Get all service titles
                $services = CreatorService::pluck('service_title');

                // Log service titles to check if they are fetched properly
                Log::info('Services:', $services->toArray());

                $words = collect();

                // Collect all words from the service titles and make them lowercase
                foreach ($services as $title) {
                    $words = $words->merge(explode(' ', strtolower($title)));
                }

                // Log words collected to check if they are split correctly
                Log::info('Words Collected:', $words->toArray());

                // Count the frequency of each word and sort by most frequent (top 5)
                $wordFrequency = $words->countBy()->sortDesc()->take(5);

                // Log word frequency to check what's being counted
                Log::info('Word Frequency:', $wordFrequency->toArray());

                // Retrieve services where the service title contains any of the most frequent words
                $recommendedServices = CreatorService::where(function ($query) use ($wordFrequency) {
                    foreach ($wordFrequency->keys() as $word) {
                        $query->orWhere('service_title', 'like', "%$word%");
                    }
                })->limit(5)->get();

                // Cache the recommendations for 1 hour (3600 seconds)
                Cache::put('recommended_services', $recommendedServices, 3600);
            }

            // If no recommendations are found, return a message
            if ($recommendedServices->isEmpty()) {
                return response()->json(['message' => 'No recommended services available.'], 404);
            }

            return response()->json(['recommended_services' => $recommendedServices]);
        }





}
