<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;

class NotificationController extends Controller
{
    public function send(Request $request)
    {
        $validated = $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'message' => 'required|string',
        ]);

        $notification = Notification::create([
            'receiver_id' => $validated['receiver_id'],
            'message' => $validated['message'],
        ]);

        return response()->json([
            'message' => 'Notification sent successfully.',
            'data' => $notification,
        ], 201);
    }

    public function index(Request $request)
    {
        $user = Auth::user();
    
        $notifications = \App\Models\Notification::where('receiver_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();
    
        $unreadCount = \App\Models\Notification::where('receiver_id', $user->id)
            ->where('is_read', false)
            ->count();
    
        return response()->json([
            'message' => 'Notifications fetched successfully.',
            'unread_count' => $unreadCount,
            'data' => $notifications
        ]);
    }

    public function markAsRead($id)
    {
        $notification = \App\Models\Notification::find($id);

        if (!$notification) {
            return response()->json([
                'message' => 'Notification not found.'
            ], 404);
        }

        $notification->is_read = true;
        $notification->save();

        return response()->json([
            'message' => 'Notification marked as read.',
            'data' => $notification
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'gig_id' => 'required|exists:gigs,id',
            'buyer_name' => 'required|string|max:255',
            'email' => 'required|email',
            'project_details' => 'required|string',
            'budget' => 'nullable|numeric|min:0',
            'expected_delivery_date' => 'required|date|after_or_equal:today',
        ]);

        $order = Order::create([
            'buyer_id' => $request->user()->id,
            'gig_id' => $request->gig_id,
            'buyer_name' => $request->buyer_name,
            'email' => $request->email,
            'project_details' => $request->project_details,
            'budget' => $request->budget,
            'expected_delivery_date' => $request->expected_delivery_date,
        ]);

        // 🔔 Get gig owner (creator)
        $gig = Gig::findOrFail($request->gig_id);
        $creator_id = $gig->user_id;

        // 📨 Save backend notification
        Notification::create([
            'sender_id' => $request->user()->id,
            'receiver_id' => $creator_id,
            'message' => 'You received a new order from ' . $request->buyer_name . '.',
        ]);

        // Optional: trigger event for real-time via Pusher (if setup)
        // event(new OrderPlacedNotification($order, $creator_id));

        return response()->json([
            'message' => 'Order placed successfully.',
            'order' => $order,
        ]);
    }

    public function getNotificationsForBuyer(Request $request)
    {
        $user = $request->user();

        $notifications = Notification::where('receiver_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $unreadCount = Notification::where('receiver_id', $user->id)
            ->where('is_read', false)
            ->count();

        return response()->json([
            'message' => 'Notifications fetched successfully.',
            'unread_count' => $unreadCount,
            'data' => $notifications,
        ]);
    }


}
